
library(ggplot2)

library(shiny)

# Define UI for application that draws a histogram
ui <- fluidPage(
    
    # Application title
    titlePanel("Visualizing uncertainty in FREM-based PVM predictions"),
    
    # Sidebar with a slider input for number of bins
    
    
    #sidebarLayout(
    #   sidebarPanel(
    fluidRow(
        column(4,
               wellPanel(
                   numericInput(inputId="AADT_baseline",
                               label="Baseline AADT",
                               value=4104),
                   numericInput(inputId="SD_AADT_baseline",
                                label="SD baseline AADT",
                                value=200),
                   numericInput(inputId="AADT_projected",
                                label="2060 projected AADT",
                                value = 21716),
                   numericInput(inputId="SD_AADT_projected",
                                label="SD projected AADT",
                                value=500),
                   numericInput(inputId="PVM_baseline",
                                label="Baseline 5-yr avg PVM",
                                value=0.4),
                   selectInput(inputId="PVM_uncert",
                                label="PVM uncertainty?",
                                choices=c("FALSE", "TRUE")),
               ),
               #plotOutput("PVMpred_Plot")
        ),
        
        
        # Show a plot of the generated distribution
        #mainPanel(
        column(8,
               plotOutput("PVMpred_Plot"),
               br(),
               #plotOutput("modelPlot"),
               #br(),
               #textOutput("Assumptions")
               # br(),
               # br(),
               # tableOutput("distSummary_p")
        )
    ),
    
    
    fluidRow(
        column(12,
               textOutput("Assumptions"))
    ),
    
    fluidRow(
        column(4,
              plotOutput("PVM_modelPlot")),
        column(4,
               plotOutput("AADTbase_modelPlot")),
        column(4,
               plotOutput("AADTproj_modelPlot"))
        
    )
)


# Define server logic required to draw a histogram
server <- function(input, output) {
    
     datasetInput <- reactive({
        nsim <- 200
        AADTbase_sim <- rnorm(nsim, mean=input$AADT_baseline, sd=input$SD_AADT_baseline)
        AADTproj_sim <- rnorm(nsim, mean=input$AADT_projected, sd=input$SD_AADT_projected)
        if(input$PVM_uncert==TRUE) {
          PVMbase_sim <- replicate(nsim, max(mean(rpois(5, lambda=input$PVM_baseline)), input$PVM_baseline))
        }
        if(input$PVM_uncert==FALSE) {
          PVMbase_sim <- rep(input$PVM_baseline, nsim)
        }
        PVMpred_sim <- (AADTproj_sim/AADTbase_sim)*PVMbase_sim
        ind_nsim <- rep(1:nsim, 2)
        PVMpred <- (input$AADT_projected/input$AADT_baseline)*input$PVM_baseline
        raw_data_df <- data.frame(Year=c(2017,2060), PVM=c(input$PVM_baseline, PVMpred))
        year <- rep(c(2017, 2060), each=nsim)
        pred_df <- data.frame(AADT_sim=c(AADTbase_sim, AADTproj_sim),
                          PVM_sim=c(PVMbase_sim, PVMpred_sim),
                          Year=year, ind_nsim=ind_nsim)
         })
    

     output$distSummary <- renderPrint({  ## SUMMARY OF UNCERTAINTY
           pred_df <- datasetInput()
           summary(pred_df$PVM[pred_df$Year==2060])
      })
    
       output$PVM_modelPlot <- renderPlot({
         if(input$PVM_uncert==TRUE) {
           PVMbase_sim <- replicate(1000, max(mean(rpois(5, lambda=input$PVM_baseline)), input$PVM_baseline))
         }
         if(input$PVM_uncert==FALSE) {
           PVMbase_sim <- rep(input$PVM_baseline, 1000)
         }
         hts <- table(PVMbase_sim)/1000
         barplot(hts, space=0.2, col=gray(0.4), 
              main="Distribution - PVM 5-yr avg",
              xlab="5-year Avg PVM")
        })
       
    output$AADTbase_modelPlot <- renderPlot({
        curve(dnorm(x, mean=input$AADT_baseline, sd=input$SD_AADT_baseline), 
              from=(input$AADT_baseline - 5*input$SD_AADT_baseline),
              to=(input$AADT_baseline + 5*input$SD_AADT_baseline),
              col="purple", lwd=2,
              ylab="Density",
              xlab="Baseline AADT",
              main="Distribution - Baseline AADT")
    })

    output$AADTproj_modelPlot <- renderPlot({
        curve(dnorm(x, mean=input$AADT_projected, sd=input$SD_AADT_projected),
              from=(input$AADT_projected - 5*input$SD_AADT_projected),
              to=(input$AADT_projected + 5*input$SD_AADT_projected),
              col="red", lwd=2,
              ylab="Density",
              xlab="Projected AADT",
              main="Distribution - projected AADT")
    })

    
    output$PVMpred_Plot <- reactivePlot( function(){
        nsim <- 500
        AADTbase_sim <- rnorm(nsim, mean=input$AADT_baseline, sd=input$SD_AADT_baseline)
        AADTproj_sim <- rnorm(nsim, mean=input$AADT_projected, sd=input$SD_AADT_projected)
        if(input$PVM_uncert==TRUE) {
          PVMbase_sim <- replicate(nsim, max(mean(rpois(5, lambda=input$PVM_baseline)), input$PVM_baseline))
        }
        if(input$PVM_uncert==FALSE) {
          PVMbase_sim <- rep(input$PVM_baseline, nsim)
        }
        PVMpred_sim <- (AADTproj_sim/AADTbase_sim)*PVMbase_sim
        ind_nsim <- rep(1:nsim, 2)
        PVMpred <- (input$AADT_projected/input$AADT_baseline)*input$PVM_baseline
        raw_data_df <- data.frame(Year=c(2017,2060), PVM=c(input$PVM_baseline, PVMpred))
        year <- rep(c(2017, 2060), each=nsim)
        pred_df <- data.frame(AADT_sim=c(AADTbase_sim, AADTproj_sim),
                              PVM_sim=c(PVMbase_sim, PVMpred_sim),
                              Year=year, ind_nsim=ind_nsim)
        p<- ggplot() +
            #geom_point(data=pred_df, aes(x=Year, y=PVM_sim, group=ind_nsim), col="blue", size=1.5, alpha=0.5) +
            geom_line(data=pred_df, aes(x=Year, y=PVM_sim, group=ind_nsim), col=gray(0.1), alpha=0.3) +
            geom_jitter(data=pred_df, aes(x=Year, y=PVM_sim, group=ind_nsim), col="blue", height=0, width=0.5, size=1.5, alpha=0.5) +
            geom_line(data=raw_data_df, aes(x=Year, y=PVM), col="cyan", lwd=1.5) +
            geom_point(data=raw_data_df, aes(x=Year, y=PVM), col="cyan", size=2, alpha=0.4) +
            ylab("5yr avg PVM")
        print(p)
    })
    
    output$Assumptions <- renderText({
        paste("Simulation from probability distributions is used to demonstrate propagation of uncertainty in FREM inputs through to PVM predictions.
        The results are based on 500 realizations (random values) from the distributions shown below.  The Poisson distribution is used for PVM counts 
        (if YES is chosen for 'PVM Uncertainty?') and normal distributions are used for both AADT inputs (with mean and standard deviation as specified
        in the input values).  The cyan line connects the baseline PVM with the point prediction from the FREM equation.")
    })
}    


#output$distPvalue <- renderText({
#    y_sim_vec <- datasetInput()
#    approx_pval <- ifelse(mean(y_sim_vec)<50, mean(y_sim_vec >= input$y_observed)*2, 
#                          mean(y_sim_vec <= input$y_observed)*2)
#    paste("Proportion as or more extreme than observed (two-tailed) = ", approx_pval)
#})
#}
# output$distSummary <- renderTable({
#     y_sim_vec <- datasetInput()
#     rbind(c("min", "0.01", "0.025", "0.05", "Median", "0.95", "0.975", "0.99", "max"),
#           round(quantile(y_sim_vec, c(0, 0.01, 0.025, 0.05, 0.5, 0.95, 0.975, 0.99, 1.0)), digits=0))},
#     colnames=FALSE, rownames=FALSE, striped=TRUE, digits=0, spacing="xs",
#     caption="Quantiles",
#     caption.placement = getOption("xtable.caption.placement", "top")
# })


# Run the application 
shinyApp(ui = ui, server = server)